﻿' =============================================================================
' [開発環境] Microsoft VisualStudio 2008 Express Edition
' [名前空間]
' [CLASS   ] Class : XlsSupport
'            Class : XlsApplication
'            Class : WorkbookCollection
'            Class : Workbook
'            Class : SheetCollection
'            Class : Sheet
'            Class : Range
'            Class : LocalFunc
'            Class : LocalComIO
' [COMMENT ] エクセルのレイトバインディング
' [CREATE  ] 2009/02/21 Nagakura
' [UPDATE  ] ----/--/--
' =============================================================================
Option Explicit On
Imports System.Reflection

#Region "XlsSupport クラス"

Public Class XlsSupport
    Implements IDisposable

    Dim mApp As XlsApplication = Nothing
    Dim mBook As Workbook = Nothing
    Dim mSheet As Sheet = Nothing
    Dim mRange As Range = Nothing

#Region "エクセル アプリケーションの操作メソッド"

    ''' ***********************************************************************
    ''' <summary>
    ''' 
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property AppCutCopyMode() As Boolean
        Get
            Return mApp.CutCopyMode
        End Get
        Set(ByVal value As Boolean)
            mApp.CutCopyMode = value
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' 警告メッセージのON・OFF
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property AppDisplayAlerts() As Boolean
        Get
            Return mApp.DisplayAlerts
        End Get
        Set(ByVal value As Boolean)
            mApp.DisplayAlerts = value
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' エクセル アプリケーションを起動する
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function AppOpen() As Boolean
        Dim bResult As Boolean = False

        Try
            AppQuit()

            mApp = New XlsApplication
            mApp.Open()

            bResult = True
        Catch
            AppQuit()
        End Try

        Return bResult
    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' エクセルアプリケーションを終了する
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub AppQuit()

        If (mRange Is Nothing) = False Then mRange.Dispose()
        mRange = Nothing

        If (mSheet Is Nothing) = False Then mSheet.Dispose()
        mSheet = Nothing

        If (mBook Is Nothing) = False Then mBook.Dispose()
        mBook = Nothing

        If (mApp Is Nothing) = False Then mApp.Quit()

        If (mApp Is Nothing) = False Then mApp.Dispose()
        mApp = Nothing

    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' Visibleプロパティを設定/解除する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property AppVisible() As Boolean
        Get
            Return mApp.Visible
        End Get
        Set(ByVal value As Boolean)
            mApp.Visible = value
        End Set
    End Property

#End Region

#Region "ブック オブジェクトの操作メソッド"

    ''' ***********************************************************************
    ''' <summary>
    ''' 新規ワークブックを開く
    ''' </summary>
    ''' <param name="TemplateFileName"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function BookAdd(Optional ByVal TemplateFileName As String = Nothing) As Boolean

        Dim bResult As Boolean = False
        Dim books As WorkbookCollection = Nothing

        Try
            If mApp Is Nothing Then
                Return False
            End If

            BookClose()

            books = mApp.Workbooks
            mBook = books.Add(TemplateFileName)
            bResult = True
        Catch
            BookClose()
        End Try

        If (books Is Nothing) = False Then
            books.Dispose()
            books = Nothing
        End If

        Return bResult

    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' ワークブックを開く
    ''' </summary>
    ''' <param name="filename"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function BookOpen(ByVal filename As String) As Boolean

        Dim bResult As Boolean = False

        Try
            If mApp Is Nothing Then
                Return False
            End If

            BookClose()

            mBook = mApp.OpenWorkbook(filename)
            bResult = True
        Catch
            BookClose()
        End Try

        Return bResult

    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' ブックを閉じる
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub BookClose()

        If (mRange Is Nothing) = False Then mRange.Dispose()
        mRange = Nothing

        If (mSheet Is Nothing) = False Then mSheet.Dispose()
        mSheet = Nothing

        If (mBook Is Nothing) = False Then mBook.Dispose()
        mBook = Nothing

    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' ブックへの変更を保存する
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub BookSave()

        If mBook Is Nothing Then
            Return
        End If

        mBook.Save()

    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' 指定されたファイル名でブックを保存する
    ''' </summary>
    ''' <param name="filename"></param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub BookSaveAs(ByVal filename As String)

        If mBook Is Nothing Then
            Return
        End If

        mBook.SaveAs(filename)

    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' 保存済みにする場合は、このプロパティをTrueにする
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property BookSaved() As Boolean
        Get
            Return mBook.Saved
        End Get
        Set(ByVal value As Boolean)
            mBook.Saved = value
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' シート数を参照する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public ReadOnly Property SheetCount()
        Get
            Dim iCountRes As Integer = 0
            Dim sheets As SheetCollection = Nothing

            Try
                sheets = mBook.Sheets()
                iCountRes = sheets.Count
                sheets.Dispose()
            Catch
                If (sheets Is Nothing) Then sheets.Dispose()
            End Try

            sheets = Nothing

            Return iCountRes
        End Get
    End Property

#End Region

#Region "シート オブジェクトの操作メソッド"

    ''' ***********************************************************************
    ''' <summary>
    ''' シートの名前を設定/参照する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property SheetName() As String
        Get
            Return mSheet.Name
        End Get
        Set(ByVal value As String)
            mSheet.Name = value
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' シートを、シートの名前を指定して選択する
    ''' </summary>
    ''' <param name="sheetname">シートの名前を指定する</param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function SheetSelect(ByVal sheetname As String)
        Dim bResult As Boolean = False

        Try
            If mApp Is Nothing Then
                Return False
            End If

            If mBook Is Nothing Then
                Return False
            End If

            SheetRelease()

            mSheet = mBook.Sheets(sheetname)

            bResult = True
        Catch
            SheetRelease()
        End Try

        Return bResult
    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' シートを、インデックスを指定して選択する
    ''' </summary>
    ''' <param name="index">インデックスを指定する</param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function SheetSelect(ByVal index As Integer)
        Dim bResult As Boolean = False

        Try
            If mApp Is Nothing Then
                Return False
            End If

            If mBook Is Nothing Then
                Return False
            End If

            SheetRelease()

            mSheet = mBook.Sheets(index)

            bResult = True
        Catch
            SheetRelease()
        End Try

        Return bResult
    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' シートの選択を解除する(シートオブジェクトの参照を解放する)
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub SheetRelease()

        If (mRange Is Nothing) = False Then mRange.Dispose()
        mRange = Nothing

        If (mSheet Is Nothing) = False Then mSheet.Dispose()
        mSheet = Nothing

    End Sub

#End Region

#Region "Range オブジェクトの操作メソッド"

    ''' ***********************************************************************
    ''' <summary>
    ''' 選択中のセルに罫線を付ける
    ''' </summary>
    ''' <param name="BorderIndex">罫線を付ける位置</param>
    ''' <param name="LineStyle">罫線の種類</param>
    ''' <param name="Weight">罫線の太さ</param>
    ''' <param name="LineColorIndex">罫線の色(1～56)</param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub RangeBorders(ByVal BorderIndex As XlBordersIndex, ByVal LineStyle As XlLineStyle, ByVal Weight As XlBorderWeight, Optional ByVal LineColorIndex As Integer = &HFFFFEFF7)
        mRange.Borders(BorderIndex, LineStyle, Weight, LineColorIndex)
    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' セルの背景色を設定/参照する(1～56)
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property RangeColorIndex() As Integer
        Get
            Return mRange.ColorIndex
        End Get
        Set(ByVal value As Integer)
            mRange.ColorIndex = value
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' 列幅を設定/参照する(標準フォントの文字数)
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property RangeColumnWidth() As Double
        Get
            Return mRange.ColumnWidth
        End Get
        Set(ByVal value As Double)
            mRange.ColumnWidth = value
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' セルをコピーする
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub RangeCopy()
        mRange.Copy()
    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' セルを切り取る
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub RangeCut()
        mRange.Cut()
    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' 行／列を挿入する。
    ''' 事前に行/列をカット,コピーしていた場合は、その内容が挿入される。
    ''' 新しい行/列を挿入する場合は事前にアプリケーションクラスのCutCopyMode=Falseを行う必要がある。
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub RangeInsert()
        mRange.Insert()
    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' Rangeのフォントを設定/取得する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property RangeFont() As Font
        Get
            Dim fn As Font = Nothing

            Try
                fn = mRange.Font
            Catch
            End Try

            Return fn
        End Get
        Set(ByVal value As Font)
            Try
                mRange.Font = value
            Catch
            End Try
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' セルの計算式を設定/取得する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property RangeFormula() As String
        Get
            Dim formulaString As String = Nothing

            Try
                formulaString = mRange.Formula
            Catch
            End Try

            Return formulaString
        End Get
        Set(ByVal value As String)
            Try
                mRange.Formula = value
            Catch
            End Try
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' 文字の表示形式に関する書式設定を設定/参照する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property RangeNumberFormatLocal() As String
        Get
            Return mRange.NumberFormatLocal
        End Get
        Set(ByVal value As String)
            mRange.NumberFormatLocal = value
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' 文字の配置-横位置を設定/参照する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property RangeHorizontalAlignment() As XlHorizontalAlignment
        Get
            Return mRange.HorizontalAlignment
        End Get
        Set(ByVal value As XlHorizontalAlignment)
            mRange.HorizontalAlignment = value
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' セルをマージする
    ''' </summary>
    ''' <param name="Across"></param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub RangeMerge(Optional ByVal Across As Boolean = False)

        If mApp Is Nothing Then
            Return
        End If

        If mBook Is Nothing Then
            Return
        End If

        If mSheet Is Nothing Then
            Return
        End If

        If mRange Is Nothing Then
            Return
        End If

        mRange.Merge(Across)

    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' 行の高さを設定/参照する(ポイント)
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property RangeRowHeight() As Double
        Get
            Return mRange.RowHeight
        End Get
        Set(ByVal value As Double)
            mRange.RowHeight = value
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' Rangeの値を設定/参照する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property RangeValue() As Object
        Get
            Dim val As Object = Nothing
            Dim range As Range = Nothing

            Try
                val = mRange.Value
            Catch
            End Try

            Return val
        End Get
        Set(ByVal value As Object)
            Try
                mRange.Value = value
            Catch
            End Try

        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' 文字の配置-縦位置を設定/参照する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property RangeVerticalAlignment() As XlVerticalAlignment
        Get
            Return mRange.VerticalAlignment
        End Get
        Set(ByVal value As XlVerticalAlignment)
            mRange.VerticalAlignment = value
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' Rangeを選択する
    ''' </summary>
    ''' <param name="rangeString"></param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub SelectRange(ByVal rangeString As String)

        Try
            If mApp Is Nothing Then
                Return
            End If

            If mBook Is Nothing Then
                Return
            End If

            If mSheet Is Nothing Then
                Return
            End If

            ReleaseRange()

            mRange = mSheet.Range(rangeString)

        Catch
            ReleaseRange()
        End Try

    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' セルを選択する
    ''' </summary>
    ''' <param name="row"></param>
    ''' <param name="col"></param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub SelectCell(ByVal row, ByVal col)

        Try
            If mApp Is Nothing Then
                Return
            End If

            If mBook Is Nothing Then
                Return
            End If

            If mSheet Is Nothing Then
                Return
            End If

            ReleaseRange()

            mRange = mSheet.Cells(row, col)

        Catch
            ReleaseRange()
        End Try

    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' 行を選択する
    ''' </summary>
    ''' <param name="row"></param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub SelectRow(ByVal row As Integer)

        Try
            If mApp Is Nothing Then
                Return
            End If

            If mBook Is Nothing Then
                Return
            End If

            If mSheet Is Nothing Then
                Return
            End If

            ReleaseRange()

            mRange = mSheet.Rows(row)

        Catch
            ReleaseRange()
        End Try

    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' 列を選択する
    ''' </summary>
    ''' <param name="col"></param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub SelectColumn(ByVal col As Integer)

        Try
            If mApp Is Nothing Then
                Return
            End If

            If mBook Is Nothing Then
                Return
            End If

            If mSheet Is Nothing Then
                Return
            End If

            ReleaseRange()

            mRange = mSheet.Columns(col)

        Catch
            ReleaseRange()
        End Try

    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' 選択しているRangeのオブジェクトを解放する
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub ReleaseRange()

        If (mRange Is Nothing) = False Then mRange.Dispose()
        mRange = Nothing

    End Sub


#End Region

    Private disposedValue As Boolean = False        ' 重複する呼び出しを検出するには

    ' IDisposable
    Protected Overridable Sub Dispose(ByVal disposing As Boolean)
        If Not Me.disposedValue Then
            If disposing Then
                ' TODO: 他の状態を解放します (マネージ オブジェクト)。
            End If

            ' TODO: ユーザー独自の状態を解放します (アンマネージ オブジェクト)。

            If (mRange Is Nothing) = False Then mRange.Dispose()
            mRange = Nothing

            If (mSheet Is Nothing) = False Then mSheet.Dispose()
            mSheet = Nothing

            If (mBook Is Nothing) = False Then mBook.Dispose()
            mBook = Nothing

            If (mApp Is Nothing) = False Then mApp.Dispose()
            mApp = Nothing

            If (mApp Is Nothing) = False Then mApp.Dispose()
            mApp = Nothing

            ' TODO: 大きなフィールドを null に設定します。
        End If
        Me.disposedValue = True
    End Sub

#Region " IDisposable Support "
    ' このコードは、破棄可能なパターンを正しく実装できるように Visual Basic によって追加されました。
    Public Sub Dispose() Implements IDisposable.Dispose
        ' このコードを変更しないでください。クリーンアップ コードを上の Dispose(ByVal disposing As Boolean) に記述します。
        Dispose(True)
        GC.SuppressFinalize(Me)
    End Sub
#End Region

End Class

#End Region

#Region "アプリケーション クラス"

''' ===========================================================================
''' <summary>
''' アプリケーション クラス
''' </summary>
''' <remarks></remarks>
''' ===========================================================================
Public Class XlsApplication
    Implements IDisposable

    ' ExcelApplication
    Private mobjAppLate As Object

    ''' ***********************************************************************
    ''' <summary>
    ''' 
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property CutCopyMode() As Boolean
        Get
            Return _
            LocalComIO.GetProperty(mobjAppLate, "CutCopyMode")
        End Get
        Set(ByVal value As Boolean)
            LocalComIO.SetProperty(mobjAppLate, "CutCopyMode", value)
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' 警告メッセージのON・OFF
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property DisplayAlerts() As Boolean
        Get
            Return _
            LocalComIO.GetProperty(mobjAppLate, "DisplayAlerts")
        End Get
        Set(ByVal value As Boolean)
            LocalComIO.SetProperty(mobjAppLate, "DisplayAlerts", value)
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' エクセルアプリケーションを起動する
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function Open() As Boolean

        Dim bResult As Boolean = False

        Try
            ' 作成するオブジェクトのプログラムID
            Dim progID As String = "Excel.Application"

            ' COMオブジェクトへの参照を作成および取得する
            Dim t As Type = Type.GetTypeFromProgID(progID)

            ' 作成されたCOMオブジェクト
            mobjAppLate = Activator.CreateInstance(t)

            If (mobjAppLate Is Nothing) = False Then
                bResult = True
            End If
        Catch
        End Try

        Return bResult

    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' ワークブックを開く
    ''' </summary>
    ''' <param name="filename"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function OpenWorkbook(ByVal filename As String) As Workbook

        Dim books As WorkbookCollection = Nothing
        Dim book As Workbook = Nothing

        Try
            ' ブックのコレクションを取得する
            books = Me.Workbooks

            ' ブック オブジェクトを取得する
            book = books.Open(filename)
        Catch
            book.Dispose()
            book = Nothing
        Finally
        End Try

        books.Dispose()
        books = Nothing

        Return book

    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' エクセルアプリケーションを終了する
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub Quit()

        If (mobjAppLate Is Nothing) = False Then
            LocalComIO.InvokeMethod(mobjAppLate, "Quit")
            LocalComIO.MRComObject(mobjAppLate)
        End If

        mobjAppLate = Nothing

    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' Visibleプロパティを設定/参照する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property Visible() As Boolean
        Get
            Return _
            DirectCast(LocalComIO.GetProperty(mobjAppLate, "Visible"), Boolean)
        End Get
        Set(ByVal value As Boolean)
            LocalComIO.SetProperty(mobjAppLate, "Visible", value)
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' ブックのコレクションを取得する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public ReadOnly Property Workbooks() As WorkbookCollection
        Get
            Dim objBooksLate As Object = Nothing

            Dim books As New WorkbookCollection

            Try
                ' workbooks のコレクションを取得する.
                books.Object = LocalComIO.GetProperty(mobjAppLate, "Workbooks")
            Catch
            End Try

            If books.Object Is Nothing Then
                books.Dispose()
                books = Nothing
            End If

            Return books
        End Get
    End Property

    Private disposedValue As Boolean = False        ' 重複する呼び出しを検出するには

    ' IDisposable
    Protected Overridable Sub Dispose(ByVal disposing As Boolean)
        If Not Me.disposedValue Then
            If disposing Then
                ' TODO: 他の状態を解放します (マネージ オブジェクト)。
            End If

            ' TODO: ユーザー独自の状態を解放します (アンマネージ オブジェクト)。
            LocalComIO.MRComObject(mobjAppLate)

            ' TODO: 大きなフィールドを null に設定します。
        End If
        Me.disposedValue = True
    End Sub

#Region " IDisposable Support "
    ' このコードは、破棄可能なパターンを正しく実装できるように Visual Basic によって追加されました。
    Public Sub Dispose() Implements IDisposable.Dispose
        ' このコードを変更しないでください。クリーンアップ コードを上の Dispose(ByVal disposing As Boolean) に記述します。
        Dispose(True)
        GC.SuppressFinalize(Me)
    End Sub
#End Region

End Class

#End Region

#Region "ブックコレクション クラス"

''' ===========================================================================
''' <summary>
''' ブックコレクション クラス
''' </summary>
''' <remarks></remarks>
''' ===========================================================================
Public Class WorkbookCollection
    Implements IDisposable

    ''' <summary>
    ''' 
    ''' </summary>
    ''' <remarks></remarks>
    Public [Object] As Object

    ''' ***********************************************************************
    ''' <summary>
    ''' 新規ワークブックを作る。
    ''' テンプレートファイル(xltファイル)を使って新しいワークブックを開く時は、そのファイル名を指定する。
    ''' </summary>
    ''' <param name="TemplateFileName">テンプレートファイル(xltファイル)パス</param>
    ''' <returns>ブック オブジェクト</returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function Add(Optional ByVal TemplateFileName As String = Nothing) As Workbook

        Dim book As Workbook = Nothing

        Try
            book = New Workbook

            If String.IsNullOrEmpty(TemplateFileName) Then
                book.Object = LocalComIO.InvokeMethod(Me.Object, "Add")
            Else
                book.Object = LocalComIO.InvokeMethod(Me.Object, "Add", TemplateFileName)
            End If
        Catch
            book.Dispose()
            book = Nothing
        End Try

        Return book

    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' ワークブックを開く
    ''' </summary>
    ''' <param name="filename"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function Open(ByVal filename As String) As Workbook

        Dim book As Workbook = Nothing

        Try
            book = New Workbook
            book.Object = LocalComIO.InvokeMethod(Me.Object, "Open", filename)
        Catch
            book.Dispose()
            book = Nothing
        End Try

        Return book

    End Function

    Private disposedValue As Boolean = False        ' 重複する呼び出しを検出するには

    ' IDisposable
    Protected Overridable Sub Dispose(ByVal disposing As Boolean)
        If Not Me.disposedValue Then
            If disposing Then
                ' TODO: 他の状態を解放します (マネージ オブジェクト)。
            End If

            ' TODO: ユーザー独自の状態を解放します (アンマネージ オブジェクト)。
            LocalComIO.MRComObject(Me.Object)

            ' TODO: 大きなフィールドを null に設定します。
        End If
        Me.disposedValue = True
    End Sub

#Region " IDisposable Support "
    ' このコードは、破棄可能なパターンを正しく実装できるように Visual Basic によって追加されました。
    Public Sub Dispose() Implements IDisposable.Dispose
        ' このコードを変更しないでください。クリーンアップ コードを上の Dispose(ByVal disposing As Boolean) に記述します。
        Dispose(True)
        GC.SuppressFinalize(Me)
    End Sub
#End Region

End Class

#End Region

#Region "ブック クラス"

''' ===========================================================================
''' <summary>
''' ブック クラス
''' </summary>
''' <remarks></remarks>
''' ===========================================================================
Public Class Workbook
    Implements IDisposable

    ' ブック オブジェクト
    Private mobjBookLate As Object

    ''' ***********************************************************************
    ''' <summary>
    ''' ブックオブジェクトを設定/参照する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property [Object]() As Object
        Get
            Return mobjBookLate
        End Get
        Set(ByVal value As Object)
            mobjBookLate = value
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' ブックを閉じる
    ''' </summary>
    ''' <param name="SaveChanges">省略可能です。ウィンドウ内のブックに変更がないとき、およびブックに変更があり、同じブックがほかのウィンドウでも表示されているとき、この引数は無視されます。ブックに変更があり、同じブックがほかのウィンドウで表示されていないとき、この引数は指定された値に応じて、動作をします。</param>
    ''' <param name="Filename">省略可能です。指定されたファイル名で、変更したファイルを保存します。</param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub Close(Optional ByVal SaveChanges As Boolean = False, Optional ByVal Filename As String = Nothing)

        If String.IsNullOrEmpty(Filename) Then
            LocalComIO.InvokeMethod(mobjBookLate, "Close", SaveChanges, Missing.Value)
        Else
            LocalComIO.InvokeMethod(mobjBookLate, "Close", SaveChanges, Filename)
        End If
    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' シートのコレクションを取得する
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public ReadOnly Property Sheets() As SheetCollection
        Get
            ' Get the worksheets collection.
            Dim objSheetsLate As Object
            objSheetsLate = LocalComIO.GetProperty(mobjBookLate, "Worksheets")

            Dim res As New SheetCollection
            res.Object = objSheetsLate

            Return res
        End Get
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' シートを、シートの名前を指定して取得する
    ''' </summary>
    ''' <param name="sheetName"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public ReadOnly Property Sheets(ByVal sheetName As String) As Sheet
        Get
            Dim sheet As Sheet = Nothing
            Dim shs As SheetCollection = Nothing

            Try
                ' シートコレクションを取得する
                shs = Me.Sheets()

                ' シートオブジェクトを取得する
                sheet = shs.Sheet(sheetName)
            Finally
                shs.Dispose()
                shs = Nothing
            End Try

            Return sheet
        End Get
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' シートを、インデックスを指定して取得する
    ''' </summary>
    ''' <param name="index"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public ReadOnly Property Sheets(ByVal index As Integer) As Sheet
        Get
            Dim sheet As Sheet = Nothing
            Dim shs As SheetCollection = Nothing

            Try
                ' シートコレクションを取得する
                shs = Me.Sheets()

                ' シートオブジェクトを取得する
                sheet = shs.Sheet(index)
            Finally
                shs.Dispose()
                shs = Nothing
            End Try

            Return sheet
        End Get
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' ブックへの変更を保存する
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub Save()
        LocalComIO.InvokeMethod(mobjBookLate, "Save")
    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' 指定されたファイル名でブックを保存する
    ''' </summary>
    ''' <param name="filename"></param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub SaveAs(ByVal filename As String)
        LocalComIO.InvokeMethod(mobjBookLate, "SaveAs", New Object() {filename})
    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' 保存済みにする場合は、このプロパティにTrueを設定する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property Saved() As Boolean
        Get
            Dim bResult As Boolean = False
            Dim objParamLate As Object = Nothing

            Try
                objParamLate = LocalComIO.GetProperty(mobjBookLate, "Saved")
                bResult = DirectCast(objParamLate, Boolean)
            Catch ex As Exception
            Finally
                If (objParamLate Is Nothing) = False Then LocalComIO.MRComObject(objParamLate)
            End Try

            Return bResult
        End Get
        Set(ByVal value As Boolean)
            LocalComIO.SetProperty(mobjBookLate, "Saved", value)
        End Set
    End Property

    Private disposedValue As Boolean = False        ' 重複する呼び出しを検出するには

    ' IDisposable
    Protected Overridable Sub Dispose(ByVal disposing As Boolean)
        If Not Me.disposedValue Then
            If disposing Then
                ' TODO: 他の状態を解放します (マネージ オブジェクト)。
            End If

            ' TODO: ユーザー独自の状態を解放します (アンマネージ オブジェクト)。
            ' TODO: 大きなフィールドを null に設定します。
        End If
        Me.disposedValue = True
    End Sub

#Region " IDisposable Support "
    ' このコードは、破棄可能なパターンを正しく実装できるように Visual Basic によって追加されました。
    Public Sub Dispose() Implements IDisposable.Dispose
        ' このコードを変更しないでください。クリーンアップ コードを上の Dispose(ByVal disposing As Boolean) に記述します。
        Dispose(True)
        GC.SuppressFinalize(Me)
    End Sub
#End Region

End Class

#End Region

#Region "シートコレクション クラス"

''' ===========================================================================
''' <summary>
''' シートコレクション クラス
''' </summary>
''' <remarks></remarks>
''' ===========================================================================
Public Class SheetCollection
    Implements IDisposable

    ''' ***********************************************************************
    ''' <summary>
    ''' 
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public [Object] As Object

    ''' ***********************************************************************
    ''' <summary>
    ''' シート数を参照する
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public ReadOnly Property Count() As Integer
        Get
            Dim iCount As Integer = 0
            Dim objCountLate As Object = Nothing

            Try
                objCountLate = LocalComIO.GetProperty(Me.Object, "Count")
                iCount = DirectCast(objCountLate, Integer)
            Finally
                LocalComIO.MRComObject(objCountLate)
            End Try

            Return iCount
        End Get
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' シートを、シートの名前を指定して取得する
    ''' </summary>
    ''' <param name="sheetName"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function Sheet(ByVal sheetName As String) As Sheet

        Dim sh As Sheet = Nothing
        Try
            sh = New Sheet
            sh.Object = LocalComIO.GetProperty(Me.Object, "Item", sheetName)
        Catch
            sh.Dispose()
            sh = Nothing
        End Try

        Return sh

    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' シートを、インデックスを指定して取得する
    ''' </summary>
    ''' <param name="index"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function Sheet(ByVal index As Integer) As Sheet

        Dim sh As Sheet = Nothing

        Try
            sh = New Sheet
            sh.Object = LocalComIO.GetProperty(Me.Object, "Item", index)
        Catch
            sh.Dispose()
            sh = Nothing
        End Try

        Return sh

    End Function

    Private disposedValue As Boolean = False        ' 重複する呼び出しを検出するには

    ' IDisposable
    Protected Overridable Sub Dispose(ByVal disposing As Boolean)
        If Not Me.disposedValue Then
            If disposing Then
                ' TODO: 他の状態を解放します (マネージ オブジェクト)。
            End If

            ' TODO: ユーザー独自の状態を解放します (アンマネージ オブジェクト)。
            LocalComIO.MRComObject(Me.Object)

            ' TODO: 大きなフィールドを null に設定します。
        End If
        Me.disposedValue = True
    End Sub

#Region " IDisposable Support "
    ' このコードは、破棄可能なパターンを正しく実装できるように Visual Basic によって追加されました。
    Public Sub Dispose() Implements IDisposable.Dispose
        ' このコードを変更しないでください。クリーンアップ コードを上の Dispose(ByVal disposing As Boolean) に記述します。
        Dispose(True)
        GC.SuppressFinalize(Me)
    End Sub
#End Region

End Class

#End Region

#Region "シート クラス"

''' ===========================================================================
''' <summary>
''' シート クラス
''' </summary>
''' <remarks></remarks>
''' ===========================================================================
Public Class Sheet
    Implements IDisposable

    ''' ***********************************************************************
    ''' <summary>
    ''' 
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public [Object] As Object

    ''' ***********************************************************************
    ''' <summary>
    ''' セルを選択する
    ''' </summary>
    ''' <param name="row"></param>
    ''' <param name="col"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function Cells(ByVal row As Integer, ByVal col As Integer) As Range

        ' エクセルシート.Cells(*,*)からRangeオブジェクトを取得する
        Dim cs As Range = Nothing

        Try
            cs = New Range
            cs.Object = LocalComIO.GetProperty(Me.Object, "Cells", New Object() {row, col})
        Catch
            cs.Dispose()
            cs = Nothing
        End Try

        Return cs

    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' 列を選択する
    ''' </summary>
    ''' <param name="col"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function Columns(ByVal col As Integer) As Range

        ' エクセルシート.Cells(*,*)からRangeオブジェクトを取得する
        Dim objColumnsLate As Object
        objColumnsLate = LocalComIO.GetProperty(Me.Object, "Columns", New Object() {col})

        Dim r As New Range
        r.Object = objColumnsLate

        Return r

    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' シートの名前を設定/参照する
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property Name() As String
        Get
            Dim sNameRes As String = Nothing
            Dim objNameLate As Object = Nothing

            Try
                objNameLate = LocalComIO.GetProperty(Me.Object, "Name")
                sNameRes = DirectCast(objNameLate, String)
            Finally
                LocalComIO.MRComObject(objNameLate)
            End Try

            Return sNameRes
        End Get
        Set(ByVal value As String)
            LocalComIO.SetProperty(Me.Object, "Name", value)
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' セルをRangeで選択する。例:Range("A1"),Range("A1:B2")
    ''' </summary>
    ''' <param name="rangeString"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function Range(ByVal rangeString As String) As Range

        ' エクセルシート.Range("**")からRangeオブジェクトを取得する
        Dim objRangeLate As Object
        objRangeLate = LocalComIO.GetProperty(Me.Object, "Range", New Object() {rangeString, Missing.Value})

        Dim r As New Range
        r.Object = objRangeLate

        Return r

    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' 行を選択する
    ''' </summary>
    ''' <param name="row"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Function Rows(ByVal row As Integer) As Range

        ' エクセルシート.Cells(*,*)からRangeオブジェクトを取得する
        Dim objRowsLate As Object
        objRowsLate = LocalComIO.GetProperty(Me.Object, "Rows", New Object() {row})

        Dim r As New Range
        r.Object = objRowsLate

        Return r

    End Function

    Private disposedValue As Boolean = False        ' 重複する呼び出しを検出するには

    ' IDisposable
    Protected Overridable Sub Dispose(ByVal disposing As Boolean)
        If Not Me.disposedValue Then
            If disposing Then
                ' TODO: 他の状態を解放します (マネージ オブジェクト)。
            End If

            ' TODO: ユーザー独自の状態を解放します (アンマネージ オブジェクト)。
            LocalComIO.MRComObject(Me.Object)

            ' TODO: 大きなフィールドを null に設定します。
        End If
        Me.disposedValue = True
    End Sub

#Region " IDisposable Support "
    ' このコードは、破棄可能なパターンを正しく実装できるように Visual Basic によって追加されました。
    Public Sub Dispose() Implements IDisposable.Dispose
        ' このコードを変更しないでください。クリーンアップ コードを上の Dispose(ByVal disposing As Boolean) に記述します。
        Dispose(True)
        GC.SuppressFinalize(Me)
    End Sub
#End Region

End Class

#End Region

#Region "Range クラス"

Public Class Range
    Implements IDisposable

    ''' ***********************************************************************
    ''' <summary>
    ''' 
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public [Object] As Object

    ''' ***********************************************************************
    ''' <summary>
    ''' 選択中のセルに罫線を付ける
    ''' </summary>
    ''' <param name="BorderIndex">罫線を付ける位置</param>
    ''' <param name="LineStyle">罫線の種類</param>
    ''' <param name="Weight">罫線の太さ</param>
    ''' <param name="LineColorIndex">罫線の色(1～56)</param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub Borders(ByVal BorderIndex As XlBordersIndex, ByVal LineStyle As XlLineStyle, ByVal Weight As XlBorderWeight, Optional ByVal LineColorIndex As Integer = &HFFFFEFF7)
        Dim objBordersLate As Object = Nothing

        Try
            ' 罫線を付ける位置を指定して、Borders オブジェクトを取得する
            objBordersLate = LocalComIO.GetProperty(Me.Object, "Borders", BorderIndex)

            ' 罫線の種類を指定する
            LocalComIO.SetProperty(objBordersLate, "LineStyle", LineStyle)

            ' 罫線の種類を指定する
            LocalComIO.SetProperty(objBordersLate, "Weight", Weight)

            ' 罫線の種類を指定する
            LocalComIO.SetProperty(objBordersLate, "ColorIndex", LineColorIndex)
        Finally
            If (objBordersLate Is Nothing) = False Then LocalComIO.MRComObject(objBordersLate)
        End Try

        objBordersLate = Nothing

    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' セル範囲をクリアする
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub Clear()
        LocalComIO.InvokeMethod(Me.Object, "Clear")
    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' セルの背景色を設定/参照する(1～56)
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property ColorIndex() As Integer
        Get
            Return _
            LocalFunc.GetColorIndex(Me.Object)
        End Get
        Set(ByVal value As Integer)
            LocalFunc.SetColorIndex(Me.Object, value)
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' 列幅を設定/参照する(標準フォントの文字数)
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property ColumnWidth() As Double
        Get
            Return _
            LocalComIO.GetProperty(Me.Object, "ColumnWidth")
        End Get
        Set(ByVal value As Double)
            LocalComIO.SetProperty(Me.Object, "ColumnWidth", value)
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' セルをコピーする
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub Copy()
        LocalComIO.InvokeMethod(Me.Object, "Copy")
    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' セルを切り取る
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub Cut()
        LocalComIO.InvokeMethod(Me.Object, "Cut")
    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' セルのフォントを設定/参照する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property Font() As System.Drawing.Font
        Get
            Return _
            LocalFunc.GetFont(Me.Object)
        End Get
        Set(ByVal value As Font)
            LocalFunc.SetFont(Me.Object, value)
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' セルに計算式を設定/参照する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property Formula() As String
        Get
            Return _
            LocalComIO.GetProperty(Me.Object, "Formula")
        End Get
        Set(ByVal value As String)
            LocalComIO.SetProperty(Me.Object, "Formula", value)
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' 行／列を挿入する。
    ''' 事前に行/列をカット,コピーしていた場合は、その内容が挿入される。
    ''' 新しい行/列を挿入する場合は事前にアプリケーションクラスのCutCopyMode=Falseを行う必要がある。
    ''' </summary>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub Insert()
        LocalComIO.InvokeMethod(Me.Object, "Insert")
    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' 文字の配置-横位置
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property HorizontalAlignment() As Integer
        Get
            Return _
            LocalComIO.GetProperty(Me.Object, "HorizontalAlignment")
        End Get
        Set(ByVal value As Integer)
            LocalComIO.SetProperty(Me.Object, "HorizontalAlignment", value)
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' Range オブジェクトの範囲のセルを結合する
    ''' </summary>
    ''' <param name="Across">省略可能です。True を設定すると、指定した範囲の各行のセルを結合して 1 つのセルにします。既定値は False です。</param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Sub Merge(Optional ByVal Across As Boolean = False)
        LocalComIO.InvokeMethod(Me.Object, "Merge", Across)
    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' 文字の表示形式に関する書式設定を設定/参照する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property NumberFormatLocal() As String
        Get
            Return _
            LocalComIO.GetProperty(Me.Object, "NumberFormatLocal")
        End Get
        Set(ByVal value As String)
            LocalComIO.SetProperty(Me.Object, "NumberFormatLocal", value)
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' 行の高さを設定/参照する(ポイント)
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property RowHeight() As Double
        Get
            Return _
            LocalComIO.GetProperty(Me.Object, "RowHeight")
        End Get
        Set(ByVal value As Double)
            LocalComIO.SetProperty(Me.Object, "RowHeight", value)
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' Rangeの値を設定/参照する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property Value() As Object
        Get
            Return _
            LocalComIO.GetProperty(Me.Object, "Value")
        End Get
        Set(ByVal value As Object)
            LocalComIO.SetProperty(Me.Object, "Value", value)
        End Set
    End Property

    ''' ***********************************************************************
    ''' <summary>
    ''' セルの文字列縦位置を設定/参照する
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Property VerticalAlignment() As Integer
        Get
            Return _
            LocalComIO.GetProperty(Me.Object, "VerticalAlignment")
        End Get
        Set(ByVal value As Integer)
            LocalComIO.SetProperty(Me.Object, "VerticalAlignment", value)
        End Set
    End Property

    Private disposedValue As Boolean = False        ' 重複する呼び出しを検出するには

    ' IDisposable
    Protected Overridable Sub Dispose(ByVal disposing As Boolean)
        If Not Me.disposedValue Then
            If disposing Then
                ' TODO: 他の状態を解放します (マネージ オブジェクト)。
            End If

            ' TODO: ユーザー独自の状態を解放します (アンマネージ オブジェクト)。
            LocalComIO.MRComObject(Me.Object)

            ' TODO: 大きなフィールドを null に設定します。
        End If
        Me.disposedValue = True
    End Sub

#Region " IDisposable Support "
    ' このコードは、破棄可能なパターンを正しく実装できるように Visual Basic によって追加されました。
    Public Sub Dispose() Implements IDisposable.Dispose
        ' このコードを変更しないでください。クリーンアップ コードを上の Dispose(ByVal disposing As Boolean) に記述します。
        Dispose(True)
        GC.SuppressFinalize(Me)
    End Sub
#End Region

End Class

#End Region

#Region "Enum定義(XlBordersIndex)"

''' <summary>
''' 罫線を付ける位置
''' </summary>
''' <remarks></remarks>
Public Enum XlBordersIndex As Integer
    ''' <summary></summary>
    xlDiagonalDown = 5
    ''' <summary></summary>
    xlDiagonalUp = 6
    ''' <summary>下</summary>
    xlEdgeBottom = 9
    ''' <summary>左</summary>
    xlEdgeLeft = 7
    ''' <summary>右</summary>
    xlEdgeRight = 10
    ''' <summary>上</summary>
    xlEdgeTop = 8
    ''' <summary>内側の横線</summary>
    xlInsideHorizontal = 12
    ''' <summary>内側の縦線</summary>
    xlInsideVertical = 11
End Enum

#End Region

#Region "Enum定義(XlBorderWeight)"

''' <summary>
''' 罫線の太さ
''' </summary>
''' <remarks></remarks>
Public Enum XlBorderWeight As Integer
    ''' <summary>極細線</summary>
    xlHairline = 1
    ''' <summary>太線</summary>
    xlMedium = &HFFFFEFD6       ' xlMedium = -4138 (&HFFFFEFD6)
    ''' <summary>極太線</summary>
    xlThick = 4
    ''' <summary>細線</summary>
    xlThin = 2
End Enum

#End Region

#Region "Enum定義(XlFileFormat)"

Public Enum XlFileFormat As Integer

    xlCSV = 6
    xlWorkbookNormal = -4143

End Enum

#End Region

#Region "Enum定義(XlHorizontalAlignment 独自定義)"

''' <summary>
''' セルの文字列横位置
''' </summary>
''' <remarks></remarks>
Public Enum XlHorizontalAlignment As Integer
    ''' <summary>標準</summary>
    xlGeneral = 1
    ''' <summary>左揃え</summary>
    xlLeft = &HFFFFEFDD         ' xlLeft = -4131 (&HFFFFEFDD)
    ''' <summary>中央揃え</summary>
    xlCenter = &HFFFFEFF4       ' xlCenter = -4108 (&HFFFFEFF4)
    ''' <summary>右揃え</summary>
    xlRight = &HFFFFEFC8        ' xlRight = -4152 (&HFFFFEFC8)
    ''' <summary>選択範囲内で中央</summary>
    xlCenterAcrossSelection = 7
    ''' <summary>均等割り付け</summary>
    xlDistributed = &HFFFFEFEB  ' xlDistributed = -4117 (&HFFFFEFEB)
End Enum

#End Region

#Region "Enum定義(XlLineStyle)"

''' <summary>
''' 罫線の種類
''' </summary>
''' <remarks></remarks>
Public Enum XlLineStyle As Integer
    ''' <summary>実線</summary>
    xlContinuous = 1
    ''' <summary>破線2</summary>
    xlDash = &HFFFFEFED             ' xlDash = -4115 (&HFFFFEFED)
    ''' <summary>破線3</summary>
    xlDashDot = 4
    ''' <summary>破線4</summary>
    xlDashDotDot = 5
    ''' <summary>破線1</summary>
    xlDot = &HFFFFEFEA              ' xlDot = -4118 (&HFFFFEFEA)
    ''' <summary></summary>
    xlDouble = &HFFFFEFE9           ' xlDouble = -4119 (&HFFFFEFE9)
    ''' <summary></summary>
    xlLineStyleNone = &HFFFFEFD2    ' xlLineStyleNone = -4142 (&HFFFFEFD2)
    ''' <summary></summary>
    xlSlantDashDot = 13
End Enum

#End Region

#Region "Enum定義(XlVerticalAlignment 独自定義)"

''' <summary>
''' セルの文字列縦位置
''' </summary>
''' <remarks></remarks>
Public Enum XlVerticalAlignment As Integer
    ''' <summary>上詰め</summary>
    xlTop = &HFFFFEFC0          ' xlTop = -4160 (&HFFFFEFC0)
    ''' <summary>中央揃え</summary>
    xlCenter = &HFFFFEFF4       ' xlCenter = -4108 (&HFFFFEFF4)
    ''' <summary>下詰め</summary>
    xlBottom = &HFFFFEFF5       ' xlBottom = -4107(&HFFFFEFF5)
    ''' <summary>均等割り付け</summary>
    xlDistributed = &HFFFFEFEB  ' xlDistributed = -4117 (&HFFFFEFEB)
End Enum

#End Region

#Region "共通"

Class LocalFunc

    ''' ***********************************************************************
    ''' <summary>
    ''' Range,Cellsからセルのフォントを取得する。
    ''' </summary>
    ''' <param name="objLate"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Shared Function GetFont(ByRef objLate As Object) As System.Drawing.Font

        Dim fontRes As Font
        Dim objFontLate As Object = Nothing
        Dim objFontNameLate As Object = Nothing
        Dim objFontSizeLate As Object = Nothing
        Dim objFontBoldLate As Object = Nothing
        Dim objFontItalicLate As Object = Nothing
        Dim objFontStrikeoutLate As Object = Nothing
        Dim objFontUnderlineLate As Object = Nothing

        Try
            ' フォントを取得する
            objFontLate = LocalComIO.GetProperty(objLate, "Font")

            ' フォントの名前を取得する
            objFontNameLate = LocalComIO.GetProperty(objFontLate, "Name")
            Dim fontName As String = DirectCast(objFontNameLate, String)

            ' フォントサイズを取得する
            objFontSizeLate = LocalComIO.GetProperty(objFontLate, "Size")
            Dim fontSize As Single = Single.Parse(objFontSizeLate.ToString())

            ' スタイル(Bold)を取得する
            objFontBoldLate = LocalComIO.GetProperty(objFontLate, "Bold")
            Dim bBold As Boolean = DirectCast(objFontBoldLate, Boolean)

            ' スタイル(Italic)を取得する
            objFontItalicLate = LocalComIO.GetProperty(objFontLate, "Italic")
            Dim bItalic As Boolean = DirectCast(objFontItalicLate, Boolean)

            ' スタイル(Strikeout:取り消し線)を取得する
            objFontStrikeoutLate = LocalComIO.GetProperty(objFontLate, "Strikethrough")
            Dim bStrikeout As Boolean = DirectCast(objFontStrikeoutLate, Boolean)

            ' スタイル(Underline)を取得する
            objFontUnderlineLate = LocalComIO.GetProperty(objFontLate, "Underline")
            Dim underline As Int32 = DirectCast(objFontUnderlineLate, Int32)

            Dim fontStyle As System.Drawing.FontStyle

            If bBold Then
                fontStyle = fontStyle Or Drawing.FontStyle.Bold
            End If

            If bItalic Then
                fontStyle = fontStyle Or Drawing.FontStyle.Italic
            End If

            If bStrikeout Then
                fontStyle = fontStyle Or Drawing.FontStyle.Strikeout
            End If

            If underline > 0 Then
                fontStyle = fontStyle Or Drawing.FontStyle.Underline
            End If

            fontRes = New Font(fontName, fontSize, fontStyle)
        Finally
            If (objFontNameLate Is Nothing) = False Then LocalComIO.MRComObject(objFontNameLate)
            If (objFontSizeLate Is Nothing) = False Then LocalComIO.MRComObject(objFontSizeLate)
            If (objFontBoldLate Is Nothing) = False Then LocalComIO.MRComObject(objFontBoldLate)
            If (objFontItalicLate Is Nothing) = False Then LocalComIO.MRComObject(objFontItalicLate)
            If (objFontUnderlineLate Is Nothing) = False Then LocalComIO.MRComObject(objFontUnderlineLate)
            If (objFontLate Is Nothing) = False Then LocalComIO.MRComObject(objFontLate)
        End Try

        Return fontRes
    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' Range,Cellsにフォントを設定する
    ''' </summary>
    ''' <param name="objLate"></param>
    ''' <param name="value"></param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Shared Sub SetFont(ByRef objLate As Object, ByRef value As Font)

        ' フォントを取得する
        Dim objFontLate As Object = Nothing

        Try
            objFontLate = LocalComIO.GetProperty(objLate, "Font")

            LocalComIO.SetProperty(objFontLate, "Name", New Object() {value.Name})
            LocalComIO.SetProperty(objFontLate, "Size", New Object() {value.Size})
            LocalComIO.SetProperty(objFontLate, "Bold", New Object() {value.Bold})
            LocalComIO.SetProperty(objFontLate, "Italic", New Object() {value.Italic})
            LocalComIO.SetProperty(objFontLate, "Strikethrough", New Object() {value.Strikeout})

            ' Excelは下線にバリエーションがある
            ' 下線無しは-4142だが、設定するとエラー発生
            If value.Underline Then
                LocalComIO.SetProperty(objFontLate, "Underline", New Object() {2})
            Else
                ' LocalComIO.SetProperty(objFontLate, "Underline", New Object() {-4142})
            End If
        Finally
            If (objFontLate Is Nothing) = False Then LocalComIO.MRComObject(objFontLate)
        End Try

    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' ColorIndexプロパティ値を取得する
    ''' </summary>
    ''' <param name="objLate"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Shared Function GetColorIndex(ByRef objLate As Object) As Integer

        Dim colorIndexRes As Integer
        Dim objInteriorLate As Object = Nothing
        Dim objColorIndex As Object = Nothing

        Try
            ' 塗りつぶし属性を取得する
            objInteriorLate = LocalComIO.GetProperty(objLate, "Interior")

            ' ColorIndexを取得する
            objColorIndex = LocalComIO.GetProperty(objInteriorLate, "ColorIndex")
            colorIndexRes = DirectCast(objColorIndex, Integer)
        Finally
            If (objInteriorLate Is Nothing) = False Then LocalComIO.MRComObject(objInteriorLate)
            If (objColorIndex Is Nothing) = False Then LocalComIO.MRComObject(objColorIndex)
        End Try

        Return colorIndexRes

    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' ColorIndexプロパティ値を設定する
    ''' </summary>
    ''' <param name="objLate"></param>
    ''' <param name="colorIndex"></param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Shared Sub SetColorIndex(ByRef objLate As Object, ByVal colorIndex As Integer)

        Dim objInteriorLate As Object = Nothing

        Try
            ' 塗りつぶし属性を取得する
            objInteriorLate = LocalComIO.GetProperty(objLate, "Interior")

            ' 設定する
            LocalComIO.SetProperty(objInteriorLate, "ColorIndex", New Object() {colorIndex})
        Finally
            If (objInteriorLate Is Nothing) = False Then LocalComIO.MRComObject(objInteriorLate)
        End Try

    End Sub

End Class

#End Region

#Region "ローカルCOMサポート クラス"

Class LocalComIO

    ''' ***********************************************************************
    ''' <summary>
    ''' オブジェクトの指定プロパティ値を取得する
    ''' </summary>
    ''' <param name="objLate">オブジェクトを指定する</param>
    ''' <param name="name">プロパティの名前を指定する</param>
    ''' <param name="Parameters"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Shared Function GetProperty(ByRef objLate As Object, ByVal name As String, ByVal ParamArray Parameters() As Object) As Object
        Return objLate.GetType().InvokeMember( _
                        name, _
                        BindingFlags.GetProperty, _
                        Nothing, _
                        objLate, _
                        Parameters)
    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' オブジェクトの指定プロパティを設定する
    ''' </summary>
    ''' <param name="objLate">オブジェクトを指定する</param>
    ''' <param name="name">プロパティの名前を指定する</param>
    ''' <param name="objValue">プロパティに設定する値を指定する</param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Shared Sub SetProperty(ByRef objLate As Object, ByVal name As String, ByVal objValue As Object)
        objLate.GetType().InvokeMember( _
                        name, _
                        BindingFlags.SetProperty, _
                        Nothing, _
                        objLate, _
                        New Object() {objValue})
    End Sub

    ''' ***********************************************************************
    ''' <summary>
    ''' メソッドを実行する
    ''' </summary>
    ''' <param name="objLate">オブジェクトを指定する</param>
    ''' <param name="name">実行するメソッドの名前を指定する</param>
    ''' <remarks>メソッドの引数を配列で指定する</remarks>
    ''' ***********************************************************************
    Public Shared Function InvokeMethod(ByRef objLate As Object, ByVal name As String, ByVal ParamArray Parameters() As Object) As Object
        Return objLate.GetType().InvokeMember( _
                        name, _
                        BindingFlags.InvokeMethod, _
                        Nothing, _
                        objLate, _
                        Parameters)
    End Function

    ''' ***********************************************************************
    ''' <summary>
    ''' COMオブジェクトの参照カウントが0になるまで
    ''' Marshal.ReleaceComObjectによるCOMオブジェクトの解放処理を行う
    ''' </summary>
    ''' <param name="objCom"></param>
    ''' <remarks></remarks>
    ''' ***********************************************************************
    Public Shared Sub MRComObject(ByRef objCom As Object)
        Try
            If objCom Is Nothing Then
                Return
            End If

            Dim i As Integer = 1
            If System.Runtime.InteropServices.Marshal.IsComObject(objCom) Then

                ' 参照カウントが0より大きい間
                Do Until i = 0
                    ' ランタイム呼び出し可能ラッパーの参照カウントをデクリメント
                    i = System.Runtime.InteropServices.Marshal.ReleaseComObject(objCom)
                Loop
            End If
        Finally
            objCom = Nothing
        End Try
    End Sub

End Class

#End Region

